/******************************************************************/
/* Title:      CPT SURGERY FLAGS LOAD SOFTWARE, VERSION 1         */
/*                                                                */
/* Program:    CPT_SURGERY_FLAGS_LOAD_PROGRAM.SAS                 */
/*                                                                */
/* Description: This SAS program creates surgery flags for each   */
/*              CPT procedure on a discharge record.  Values of   */
/*              '1'=BROAD, '2'=NARROW, and '0'=NEITHER will       */
/*              populate each flag. 	                          */
/*                                                                */
/*              Please note that in order to build a list of all  */
/*              discharges with "BROAD" surgeries, analysts will  */
/*              need to capture codes with surgery flag=1 (BROAD) */
/*              as well as codes that have surgery flag=2 (NARROW)*/
/*              because the narrow codes are also broad by        */
/*              definition. See sample code in lines 71-78.       */
/*                                                                */
/*              The program is intended for data with CPT         */
/*              procedure codes. The CPT tool file only contains  */
/*              flags for CPT codes that are considered surgical. */
/*              These codes were identified from the AMA surgical */
/*              range (10021-69999).                              */
/*              Any CPT codes that are not in the tool files are  */
/*              assigned a surgery flag of 0.                     */
/*                                                                */
/*              The program assumes the use of discharge-level    */
/*              data where a single discharge record contains all */
/*              of the procedure codes on a single line.          */
/*                                                                */
/*              Users must indicate if their data records         */
/*              contain CPT codes by specifying the maximum number*/
/*              of CPT codes found on their records. For example, */
/*              there are four CPT codes on each record, the      */
/*              maximum number of CPT codes flag (NUMCPTPRS)      */
/*              is set to 4 below. If there are no CPT            */
/*              codes on any record, NUMCPTPRS is set to 0 below. */
/*              See line 85 below to update.                      */
/*                                                                */
/*              There are two general sections to this program:   */
/*                                                                */
/*              1) The first section creates a temporary SAS      */
/*                 format using the surgery flag ".csv"           */
/*                 file. This formats is used in step two to      */
/*                 create the surgery flag variables.             */
/*                                                                */
/*              2) The second section loops through the CPT       */
/*                 procedure array on your SAS dataset            */
/*                 and creates a set or sets of surgery flags     */
/*                 from the procedures present.                   */ 
/*                                                                */
/*                 Users should update lines 85-86, 91-93,        */
/*                 144, 154-155, 172                              */
/*                                                                */
/* Optional Flags: Users may wish to create a record-level flag   */
/*                 to identify the entire discharge as surgical or*/
/*                 not. In order to do this, analysts should scan */
/*                 the CPT surgery flags created by               */
/*                 the program below.  If any surgery flag on a   */
/*                 record contains the values '1' or '2', two     */
/*                 record-level surgery flags would be created    */
/*                 with a possible value of 0 for not surgical    */
/*                 or 1 for surgical.                             */
/*                 If users would like a record-level flag of     */
/*                 surgical procedures, broad_surgery=1 can be    */
/*                 used since it fundamentally includes both broad*/
/*                 and narrow surgical procedures.                */
/*                                                                */
/*                 An example of SAS code to do this which could  */
/*                 be added at the end of the program follows:    */
/*                                                                */
//*	           %if &NUMCPTPRS > 0 %then %do;                      */
/*                    DO I = 1 TO &NUMCPTPRS;                     */
/*                       IF CPTSFLAGS(I) = '2' THEN               */
/*                       NARROW_SURGERY = 1;                      */
/*                       IF CPTFLAGS(I) IN('1','2') THEN          */
/*                       BROAD_SURGERY = 1;                       */
/*                    END;                                        */
/*	           %end;                                              */

/******************************************************************/

***************************************************************;
*** THE SAS MACRO FLAGS BELOW MUST BE UPDATED BY THE USER *****;
***************************************************************;
* SPECIFY THE NUMBER OF CPT PROCEDURES ON YOUR RECORDS;      %LET NUMCPTPRS=4;
* SPECIFY YOUR SAS FILE NAME;                                %LET CORE=testcpt;

***********************************************************************************;
*** THE SAS PATHS BELOW MUST BE UPDATED BY THE USER TO MATCH YOUR FILE SYSTEM *****;
***********************************************************************************;
FILENAME INRAW1  'C:\users\u5930421\desktop\TOOLS\SURGERY_FLAGS_CPT_2019.CSV' LRECL=300;  
LIBNAME  IN1     'C:\users\u5930421\desktop\TOOLS\';                 * Location of input discharge data;
LIBNAME  OUT1    'C:\users\u5930421\desktop\TOOLS\';                 * Location of output data;


TITLE1 'CREATE SURGERY FLAGS';
TITLE2 'USE WITH DISCHARGE ADMINISTRATIVE DATA CONTAINING CPT PROCECDURE CODES';

options mprint symbolgen;

%Macro surgery_flags;
/******************* SECTION 1: CREATE INFORMATS *******************/
/*  SAS Load the CPT surgery flag tool and convert it into a       */
/*  temporary SAS format. The format will be used in the data step */
/*  below to assign the surgery flags for each  procedure on a     */
/*  record.                                                        */
/*******************************************************************/

DATA CPT_SURGERY_FLAGS;
   INFILE INRAW1 DSD DLM=',' END = EOF FIRSTOBS=3;
	INPUT
	   RANGE       : $CHAR13.
	   LABEL       : $1.
	;

   END   = SCAN(RANGE,2);
   START = SCAN(RANGE,1);
   RETAIN HLO " ";
   FMTNAME = "$CPsf" ;
   TYPE    = "C" ;
   OUTPUT;

   IF EOF THEN DO ;
      START = " " ;
	  END   = " " ;
      LABEL = "0" ;
      HLO   = "O";
      OUTPUT ;
   END ;
RUN;

PROC FORMAT LIB=WORK CNTLIN = CPT_SURGERY_FLAGS;
RUN;


/********** SECTION 2: CREATE SURGERY FLAGS VARIABLES *************/
/*  Create surgery flag variables for each CPT code using the SAS */
/*  format created above and the SAS file you wish to augment.    */
/*  Users can change the names of the output surgery flag         */
/*  variables if needed here. It is also important to make sure   */
/*  that the correct CPT procedure names from your SAS file are   */
/*   used in the array 'CPTPRS'.                                  */
/******************************************************************/  
DATA OUT1.NEW_SURGERY_FLAGS (DROP = i);			    * The SAS output dataset name can be changed here;
  SET IN1.&CORE;

  /***************************************************/
  /*  First build arrays for all of the surgery      */
  /*  flag variables that are going to be created.   */
  /*  Also build arrays that contain the CPT codes   */
  /*  on the input file.                             */
  /***************************************************/
  %if &NUMCPTPRS > 0 %then %do;
     ARRAY CPTSFLAGS (*)   $1 SURG_CPT_FLG1-SURG_CPT_FLG&NUMCPTPRS;   * Suggested name for the CPT minor therapeutic and diagnostic surgery flags variables;
     ARRAY CPTPRS    (*)   $5 CPT1-CPT&NUMCPTPRS;                     * Change CPT procedure variable names to match your file;
  %end;
 

  /***************************************************/
  /*  Loop through the CPT procedure codes found     */ 
  /*  on your SAS input dataset and create a surgery */
  /*  flag variable for each code. This step makes   */
  /*  use of the CPT arrays above.                   */
  /***************************************************/
  %if &NUMCPTPRS > 0 %then %do;
     DO I = 1 TO &NUMCPTPRS;
	     CPTSFLAGS(I) = PUT(CPTPRS(I),$CPsf.);
     END;  
  %end;
RUN;

PROC PRINT DATA=OUT1.NEW_SURGERY_FLAGS   (OBS=10);  * If the SAS output dataset name changed above, make sure to change it here;
   VAR 
       %if &NUMCPTPRS > 0 %then %do;CPT1 SURG_CPT_FLG1 CPT2 SURG_CPT_FLG2%end;
		 ;
  title3 "PARTIAL PRINT OF THE SURGERY FLAGS FILE";
RUN;
%Mend surgery_flags;
%surgery_flags;



